<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Not authorized']);
    exit;
}

try {
    if (!isset($_GET['sale_id'])) {
        throw new Exception('Sale ID is required');
    }

    $sale_id = $_GET['sale_id'];

    // Get sale items with product details and sale information
    $stmt = $pdo->prepare("
        SELECT 
            si.id,
            si.product_id,
            p.name,
            si.quantity,
            si.price as unit_price,
            si.discount,
            s.payment_method,
            s.total_amount as sale_total,
            s.created_at as sale_date,
            GROUP_CONCAT(DISTINCT psn.serial_number ORDER BY psn.created_at ASC) as serial_numbers
        FROM sale_items si
        JOIN products p ON si.product_id = p.id
        JOIN sales s ON si.sale_id = s.id
        LEFT JOIN product_serial_numbers psn ON p.id = psn.product_id 
            AND psn.status = 'sold' 
            AND psn.source_id = si.sale_id
            AND (
                (psn.source_type = 'manual_adjustment')
                OR 
                (psn.source_type = 'credit_sale')
            )
        WHERE si.sale_id = ?
        GROUP BY si.id, si.product_id, p.name, si.quantity, si.price, si.discount, s.payment_method, s.total_amount, s.created_at
    ");
    $stmt->execute([$sale_id]);
    $items = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get multiple payment methods from sale_payments table
    $stmt = $pdo->prepare("
        SELECT method, amount 
        FROM sale_payments 
        WHERE sale_id = ? 
        ORDER BY created_at ASC
    ");
    $stmt->execute([$sale_id]);
    $payment_methods = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get refund information for this sale
    $stmt = $pdo->prepare("
        SELECT 
            r.id as refund_id,
            r.amount as refund_amount,
            r.status as refund_status,
            COUNT(ri.id) as refunded_items_count,
            SUM(ri.quantity) as total_refunded_quantity
        FROM refunds r
        LEFT JOIN refund_items ri ON r.id = ri.refund_id
        WHERE r.sale_id = ?
        GROUP BY r.id, r.amount, r.status
    ");
    $stmt->execute([$sale_id]);
    $refunds = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Get which specific items have been returned
    $stmt = $pdo->prepare("
        SELECT 
            ri.sale_item_id,
            ri.product_id,
            ri.quantity as returned_quantity
        FROM refund_items ri
        JOIN refunds r ON ri.refund_id = r.id
        WHERE r.sale_id = ?
    ");
    $stmt->execute([$sale_id]);
    $returned_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    if (!isset($returned_items) || !is_array($returned_items)) {
        $returned_items = [];
    }

    // Calculate total items and quantities for refund status
    $total_items = 0;
    $total_quantity = 0;
    foreach ($items as $item) {
        $total_items++;
        $total_quantity += $item['quantity'];
    }

    $total_refunded_items = 0;
    $total_refunded_quantity = 0;
    foreach ($refunds as $refund) {
        $total_refunded_items += $refund['refunded_items_count'];
        $total_refunded_quantity += $refund['total_refunded_quantity'];
    }

    // Determine refund status
    $refund_status = 'none';
    if ($total_refunded_items > 0) {
        if ($total_refunded_items >= $total_items && $total_refunded_quantity >= $total_quantity) {
            $refund_status = 'complete';
        } else {
            $refund_status = 'partial';
        }
    }

    // Format the response
    $formatted_items = array_map(function($item) use ($returned_items, $sale_id, $pdo) {
        // Parse serial numbers from concatenated string to array
        $serial_numbers = [];
        if (!empty($item['serial_numbers'])) {
            $serial_array = explode(',', $item['serial_numbers']);
            foreach ($serial_array as $serial) {
                $serial = trim($serial);
                if (!empty($serial)) {
                    $serial_numbers[] = ['serial_number' => $serial];
                }
            }
        }
        
        // Check 7-day return policy
        $sale_date = new DateTime($item['sale_date']);
        $current_date = new DateTime();
        $days_difference = $current_date->diff($sale_date)->days;
        $can_return = $days_difference <= 7;
        
        // Check if this item has been returned
        $returned_quantity = 0;
        $is_returned = false;
        foreach ($returned_items as $returned_item) {
            if ($returned_item['sale_item_id'] == $item['id']) {
                $returned_quantity = $returned_item['returned_quantity'];
                $is_returned = ($returned_quantity >= $item['quantity']);
                break;
            }
        }
        
        // Calculate remaining quantity that can be returned
        $remaining_quantity = $item['quantity'] - $returned_quantity;
        $can_return_item = $can_return && $remaining_quantity > 0;
        
        // Get available serial numbers for selection (only for items with serial numbers)
        $available_serials = [];
        if (!empty($serial_numbers)) {
            // Get serial numbers that are still sold (not yet returned)
            $stmt = $pdo->prepare("
                SELECT serial_number 
                FROM product_serial_numbers 
                WHERE product_id = ? 
                AND source_type IN ('manual_adjustment', 'credit_sale')
                AND source_id = ?
                AND status = 'sold'
                ORDER BY created_at ASC
            ");
            $stmt->execute([$item['product_id'], $sale_id]);
            $available_serials = $stmt->fetchAll(PDO::FETCH_COLUMN);
        }
        
        return [
            'id' => $item['id'],
            'product_id' => $item['product_id'],
            'name' => $item['name'],
            'quantity' => $item['quantity'],
            'unit_price' => floatval($item['unit_price']),
            'discount' => floatval($item['discount'] ?? 0),
            'serial_numbers' => $serial_numbers,
            'sale_date' => $item['sale_date'],
            'days_since_sale' => $days_difference,
            'can_return' => $can_return_item,
            'return_policy_message' => $can_return_item ? 'Eligible for return' : ($is_returned ? 'Already returned' : 'Return period expired (7 days)'),
            'returned_quantity' => $returned_quantity,
            'remaining_quantity' => $remaining_quantity,
            'is_returned' => $is_returned,
            'available_serials' => $available_serials,
            'has_serials' => !empty($serial_numbers)
        ];
    }, $items);

    // Get primary payment method from sales table (for backward compatibility)
    $primary_payment_method = !empty($items) ? ucfirst($items[0]['payment_method']) : 'Cash';
    $sale_total = !empty($items) ? floatval($items[0]['sale_total']) : 0;
    $sale_date = !empty($items) ? $items[0]['sale_date'] : null;

    // Calculate total paid by each method (credit is debt, not payment)
    $payment_breakdown = [];
    $total_paid = 0;
    $credit_amount = 0;
    foreach ($payment_methods as $payment) {
        $method = strtolower($payment['method']);
        $amount = floatval($payment['amount']);
        
        if ($method === 'credit') {
            // Credit is debt, not payment - track separately
            $credit_amount = $amount;
            $payment_breakdown[] = [
                'method' => 'Credit',
                'amount' => $amount,
                'is_debt' => true
            ];
        } else {
            // Real payments (cash, mpesa, etc.)
            $payment_breakdown[] = [
                'method' => ucfirst($method),
                'amount' => $amount,
                'is_debt' => false
            ];
            $total_paid += $amount;
        }
    }

    // If no payment methods found, use the primary payment method
    if (empty($payment_breakdown)) {
        $is_credit = strtolower($primary_payment_method) === 'credit';
        $payment_breakdown[] = [
            'method' => $primary_payment_method,
            'amount' => $sale_total,
            'is_debt' => $is_credit
        ];
        if (!$is_credit) {
            $total_paid = $sale_total;
        }
    }

    echo json_encode([
        'success' => true,
        'items' => $formatted_items,
        'payment_method' => $primary_payment_method,
        'payment_methods' => $payment_breakdown,
        'sale_total' => $sale_total,
        'total_paid' => $total_paid,
        'credit_amount' => $credit_amount,
        'has_multiple_payments' => count($payment_breakdown) > 1,
        'sale_date' => $sale_date,
        'refund_status' => $refund_status,
        'total_items' => $total_items,
        'total_quantity' => $total_quantity,
        'total_refunded_items' => $total_refunded_items,
        'total_refunded_quantity' => $total_refunded_quantity,
        'can_return_any' => array_reduce($formatted_items, function($carry, $item) {
            return $carry || $item['can_return'];
        }, false)
    ]);

} catch (Exception $e) {
    error_log("Get Sale Items Error: " . $e->getMessage());
    echo json_encode([
        'success' => false,
        'message' => 'Error fetching sale items: ' . $e->getMessage()
    ]);
} 